#include "ados.h"
#include "standard.h"

/* the CATALOG_CLICK is the increment size for adding file entries	*/
/* if it is a low number, then I forgot to increase it after testing!	*/

#define CATALOG_CLICK	5

/* the number of file entries in each catalog sector	*/

#define CATALOG_FILES_PER_SECTOR	7

static ados_file_entry	blank_file_entry;

/************************************************************************
 * NAME:	ados_create_blank_file_entry()
 *
 * DESCR:	Creates a blank ados_file_entry which is used when clearing
 *		out catalog entries upon write.  The static "blank_file_entry"
 *		is filled in.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
static void
ados_create_blank_file_entry()
{
    blank_file_entry.ts.track = 0;
    blank_file_entry.ts.sector = 0;
    blank_file_entry.locked = FALSE;
    blank_file_entry.deleted = FALSE;

    {
	int	i;
	for (i=0; i < ADOS_FILE_NAME_LENGTH; i++) {
	    blank_file_entry.name[i] = ' ';
	}
    }
}



/************************************************************************
 * NAME:	catalog_expand()
 *
 * DESCR:	Expands the catalog storage space.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it went OK, FALSE otherwise.
 *
 * NOTES:	
 ************************************************************************/
static int
ados_catalog_expand(ados_catalog *cat)
{
    ados_file_entry	*new_array;

    new_array = (ados_file_entry *)realloc(cat->files,sizeof(ados_file_entry)*(cat->count+CATALOG_CLICK));
    if (new_array) {
	cat->files = new_array;
	cat->count += CATALOG_CLICK;
	return(TRUE);
    } else {
	return(FALSE);
    }
}

/************************************************************************
 * NAME:	ados_catalog_destroy()
 *
 * DESCR:	Destroys the given catalog.  It frees the file entries
 *		that are part of the catalog as well.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
ados_catalog_destroy(ados_catalog *cat)
{
    free(cat->files);
    free(cat);
}

/************************************************************************
 * NAME:	ados_catalog_create()
 *
 * DESCR:	Creates a catalog structure and prepares for the file
 *		entries that will be in it.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
ados_catalog *
ados_catalog_create()
{
    ados_catalog	*retcat = NULL;

    if ((retcat = (ados_catalog *)malloc(sizeof(ados_catalog))) != NULL) {
	retcat->sectors = 0;
	retcat->index = 0;
	retcat->count = 0;
	retcat->files = NULL;
	if (!ados_catalog_expand(retcat)) {
	    ados_catalog_destroy(retcat);
	    retcat = NULL;
	}
    }

    return(retcat);
}

/************************************************************************
 * NAME:	file_entry_read()
 *
 * DESCR:	Given a data pointer, "read" out a file entry structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- the file_entry that is returned is to STATIC space
 *		  ie - do not count on it remaining when called again.
 ************************************************************************/
ados_file_entry *
file_entry_read(unsigned char *data)
{
    static ados_file_entry	newfile;

    newfile.ts.track = data[0x00];
    newfile.ts.sector = data[0x01];
    newfile.length = int_read(&(data[0x21]));
    newfile.deleted = FALSE;

    if (newfile.ts.track == 0) {	/* no file in this entry	*/
	return(NULL);
    }

    if (newfile.ts.track == 0xff) {	/* deleted entry here		*/
	newfile.ts.track = data[0x20];	/* hidden track number here	*/
	newfile.deleted = TRUE;
	data[0x20] = ' ';			/* put a space in there	*/
    }

    newfile.locked = FALSE;
    if (data[0x02] & 0x80) {
	newfile.locked = TRUE;
    }

    /* this isn't a switch/case in that I'm not confident that the 	*/
    /* data in 0x02 has EXACTLY one bit set...using these if's, then at	*/
    /* least 1 type will be set, maybe more.				*/

    if (data[0x02] & 0x01) {
	newfile.type = ADOS_FILE_IBASIC;
    } else if (data[0x02] & 0x02) {
	newfile.type = ADOS_FILE_ABASIC;
    } else if (data[0x02] & 0x04) {
	newfile.type = ADOS_FILE_BIN;
    } else if (data[0x02] & 0x08) {
	newfile.type = ADOS_FILE_S;
    } else if (data[0x02] & 0x10) {
	newfile.type = ADOS_FILE_R;
    } else if (data[0x02] & 0x20) {
	newfile.type = ADOS_FILE_A;
    } else if (data[0x02] & 0x40) {
	newfile.type = ADOS_FILE_B;
    } else {
	newfile.type = ADOS_FILE_TEXT;
    }

    {
	/* it appears that the upper bit is set in DSK images for the	*/
	/* files.  I wonder if it NEEDS to be set (ie for write)	*/

	int	i;
	for (i=0; i < ADOS_FILE_NAME_LENGTH; i++) {
	    newfile.name[i] = data[0x03+i] & 0x7f;
	}
    }

    return(&newfile);
}

/************************************************************************
 * NAME:	file_entry_write()
 *
 * DESCR:	Write out the given file entry to the supplied spot.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
file_entry_write(unsigned char *data, ados_file_entry *file_entry)
{
    data[0x00] = file_entry->ts.track;
    data[0x01] = file_entry->ts.sector;
    int_write(file_entry->length,&(data[0x21]));

    data[0x02] = 0;
    if (file_entry->locked) {
	data[0x02] |= 0x80;
    }

    switch(file_entry->type) {
        case ADOS_FILE_IBASIC:	data[0x02] |= 0x01; break;
	case ADOS_FILE_ABASIC:	data[0x02] |= 0x02; break;
	case ADOS_FILE_BIN:	data[0x02] |= 0x04; break;
	case ADOS_FILE_S:	data[0x02] |= 0x08; break;
	case ADOS_FILE_R:	data[0x02] |= 0x10; break;
	case ADOS_FILE_A:	data[0x02] |= 0x20; break;
	case ADOS_FILE_B:	data[0x02] |= 0x40; break;
        case ADOS_FILE_TEXT:	break;		/* nothing extra for text 	*/
    }

    {
	/* it appears that the upper bit is set in DSK images for the	*/
	/* files.  Go ahead and set it here too.			*/

	int	i;
	for (i=0; i < ADOS_FILE_NAME_LENGTH; i++) {
	    data[0x03+i] = file_entry->name[i] | 0x80;
	}
    }

    if (file_entry->deleted) {
	data[0x00] = 0xff;
	data[0x20] = file_entry->ts.track;	/* hidden track here	*/
    }
}

/************************************************************************
 * NAME:	ados_catalog_add_file
 *
 * DESCR:	Adds a file to the catalog.  Expands storage as necessary.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
ados_catalog_add_file(ados_catalog *cat, ados_file_entry *file) 
{
    /* check to see if we have enough space for more...	*/

    if (cat->index >= (CATALOG_FILES_PER_SECTOR * cat->sectors)) {
	return(FALSE);
    }

    if (cat->index == cat->count) {
	if (!ados_catalog_expand(cat)) {
	    return(FALSE);
	}
    }

    /* structure copy here	*/

    cat->files[cat->index++] = *file;

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_catalog_read()
 *
 * DESCR:	Read in a catalog from the given file system.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- assumes that the VTOC is set-up already
 *		- a call to "ados_create_blank_file_entry()" is done so
 *		  to initialize the blank entry for use when writing.
 ************************************************************************/
ados_catalog *
ados_catalog_read(struct adosfs *adosfs)
{
    ados_catalog		*cat;
    unsigned char	 data[CATALOG_SECTOR_SIZE];
    int			 track, sector;
    int			 i;

    ados_create_blank_file_entry();

    /* first allocate the storage for the catalog	*/

    if ((cat = ados_catalog_create()) == NULL) {
	return(NULL);
    }
       
    /* get the first sector where the catalog starts	*/

    track = adosfs->vtoc->catalog.track;
    sector = adosfs->vtoc->catalog.sector;

    while (track != 0) {
	if (!ados_getlsect(adosfs, track, sector, data)) {
	    ados_catalog_destroy(cat);
	    return(NULL);
	}

	cat->sectors++;

	track = data[0x01];
	sector = data[0x02];

	for (i=0; i < CATALOG_FILES_PER_SECTOR; i++) {
	    ados_file_entry *newfile;
	    newfile = file_entry_read(&(data[0x0b+(i*FILE_ENTRY_SIZE)]));

	    if (newfile && !ados_catalog_add_file(cat,newfile)) {
		ados_catalog_destroy(cat);
		return(NULL);
	    }
	}
    }

    return(cat);
}

/************************************************************************
 * NAME:	ados_catalog_write()
 *
 * DESCR:	Re-writes the current catalog to the disk image.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if OK, FALSE otherwise
 *
 * NOTES:	- the current catalog on the image is completely replaced.
 *		- how do we reuse the existing catalog sectors?
 *		- it should start at track 17, using F through 1, and
 *		  "reserving" that space too.
 ************************************************************************/
int
ados_catalog_write(struct adosfs *adosfs)
{
    ados_catalog	*cat;
    unsigned char	 data[CATALOG_SECTOR_SIZE];
    int			 track, sector;
    int			 next_track, next_sector;
    int			 i;
    int			 cat_entry;

    /* writing the catalog involves writing each of the sectors	*/
    /* in the catalog space.  Normally these go from 15 down to	*/
    /* one.  So the algorithm here does the following:		*/
    /*	- go to the first catalog sector that used to be there	*/
    /*	- start writing catalog sectors with the file names	*/
    /*    packed toward the beginning.				*/
    /*	- when done with the files, clear the remaining catalog	*/
    /*	  sectors, but leave them allocated.			*/

    /* get the first sector where the catalog starts	*/

    cat = adosfs->catalog;

    track = adosfs->vtoc->catalog.track;
    sector = adosfs->vtoc->catalog.sector;

    cat_entry = 0;

    while (track != 0) {

	/* need to get this sector so that we can see where the	*/
	/* next catalog sector is located...then rewrite it.	*/

	if (!ados_getlsect(adosfs, track, sector, data)) {
	    return(FALSE);
	}

	for (i=0; i < CATALOG_FILES_PER_SECTOR; i++) {
	    if (cat_entry >= cat->index) {
		break;				/* no more to be written	*/
	    }
	    file_entry_write(&(data[0x0b+(i*FILE_ENTRY_SIZE)]), &(cat->files[cat_entry]));
	    cat_entry++;
	}

	for ( ; i < CATALOG_FILES_PER_SECTOR; i++) {	/* left over blanks to be cleared	*/
	    file_entry_write(&(data[0x0b+(i*FILE_ENTRY_SIZE)]), &blank_file_entry);
	}

	if (!ados_putlsect(adosfs, track, sector, data)) {
	    return(FALSE);
	}

	track = data[0x01];
	sector = data[0x02];
    }

    return(TRUE);
}

/************************************************************************
 * NAME:	ados_catalog_delete()
 *
 * DESCR:	Deletes the given file from the filesystem.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
ados_catalog_delete(ados_file_entry *entry)
{
    entry->deleted = TRUE;
}

/************************************************************************
 * NAME:	ados_catalog_map_extension()
 *
 * DESCR:	Map the given extension to an ados filetype.  Note that
 *		the extension can be given as null.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
ados_file_type
ados_catalog_map_extension(char *extension)
{
    ados_file_type	filetype;

         if (extension == NULL)			filetype = ADOS_FILE_BIN;
    else if (strcasecmp(extension,"bin") == 0 )	filetype = ADOS_FILE_BIN;
    else if (strcasecmp(extension,"text") == 0 )filetype = ADOS_FILE_TEXT;
    else if (strcasecmp(extension,"txt") == 0 )	filetype = ADOS_FILE_TEXT;
    else if (strcasecmp(extension,"bas") == 0 ) filetype = ADOS_FILE_IBASIC;
    else if (strcasecmp(extension,"ibas") == 0 )filetype = ADOS_FILE_IBASIC;
    else if (strcasecmp(extension,"abas") == 0 )filetype = ADOS_FILE_ABASIC;
    else if (strcasecmp(extension,"s") == 0 )	filetype = ADOS_FILE_S;
    else if (strcasecmp(extension,"r") == 0 )	filetype = ADOS_FILE_R;
    else if (strcasecmp(extension,"rel") == 0 )	filetype = ADOS_FILE_R;
    else if (strcasecmp(extension,"a") == 0 )	filetype = ADOS_FILE_A;
    else if (strcasecmp(extension,"b") == 0 )	filetype = ADOS_FILE_B;
    else 					filetype = ADOS_FILE_BIN;

    return(filetype);
}

/************************************************************************
 * NAME:	ados_catalog_map_filetype()
 *
 * DESCR:	Maps the given filetype into a string used to create a
 *		standard filename.
 *
 * ARGS:	
 *
 * RETURNS:	returns pointers to static areas that should be copied.
 *
 * NOTES:	
 ************************************************************************/
char *
ados_catalog_map_filetype(ados_file_type filetype)
{
    switch(filetype) {
    case ADOS_FILE_BIN:		return("bin");
    case ADOS_FILE_TEXT:	return("txt");
    case ADOS_FILE_IBASIC:	return("ibas");
    case ADOS_FILE_ABASIC:	return("abas");
    case ADOS_FILE_S:		return("s");
    case ADOS_FILE_R:		return("r");
    case ADOS_FILE_A:		return("a");
    case ADOS_FILE_B:		return("b");
    }
}

/************************************************************************
 * NAME:	ados_catalog_translate() (IN and OUT)
 *
 * DESCR:	Translate a "modern" file-name to an ADOS filename, and
 *		vice-versa.
 *		NOTE - if the given filename is greater than will fit in
 *		the ADOS filename structure, then it is truncated to fit.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- this means setting the file type with the extension
 *		- trim any path name stuff
 *		- padding the file with spaces on the end
 *		- all upper case
 *		- returned file is in a static space and should be copied
 *		- assumes the incoming filename is terminated with '\0'
 ************************************************************************/
char *
ados_catalog_translate_in(char *input, ados_file_type *filetype)
{
    static char		buffer[ADOS_FILE_NAME_LENGTH];
    char		*extension, *basename;
    int			i, count;

    extern char *	floppy_file_basename(char *);
    extern char *	floppy_file_extension(char *);

    basename = floppy_file_basename(input);		/* ptr to where basename starts	*/
    extension = floppy_file_extension(basename);	/* ptr to where ext starts	*/

    *filetype = ados_catalog_map_extension(extension);

    if (extension == NULL) {
	count = MIN(strlen(basename),ADOS_FILE_NAME_LENGTH);
    } else {
	count = MIN(extension - basename - 1,ADOS_FILE_NAME_LENGTH);
    }

    for (i=0; i < count; i++) {
	buffer[i] = toupper(basename[i]);
    }

    for ( ; i < ADOS_FILE_NAME_LENGTH; i++) {	/* space fill remainder	*/
	buffer[i] = ' ';
    }
    
    return(buffer);
}

char *
ados_catalog_translate_out(char *input, ados_file_type filetype)
{
    /** NOTE:	- the +6 in the buffer definition is meant to account	*/
    /**		  the dot and extension, as well as a null termination.	*/
    /**		  This assumes that the biggest ext used is 4 chars.	*/

    static char		buffer[ADOS_FILE_NAME_LENGTH+6];
    int			i, j;
    char		*newext;

    /* scan for the first trailing non-blank	*/

    for (i=ADOS_FILE_NAME_LENGTH; i--; ) {
	if (input[i] != ' ') {
	    break;
	}
    }

    /* copy the filename first	*/

    for (j=0; j <= i; j++) {
	buffer[j] = tolower(input[j]);
    }
    
    buffer[j++] = '.';		/* always put an extension on output filenames	*/

    /* add the extension to match the file type	*/

    newext = ados_catalog_map_filetype(filetype);

    for (i=0; i < strlen(newext); i++) {
	buffer[j++] = newext[i];
    }

    buffer[j] = '\0';		/* null termination	*/

    return(buffer);
}

/************************************************************************
 * NAME:	ados_catalog_new()
 *
 * DESCR:	Creates a new entry in the catalog, which equates to
 *		creating a new file.  Note that the TS sector is allocated
 *		for the new file immediately.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- according to the rules, catalog entries can be created
 *		  until the cow's come home :-).  Due to the file structure
 *		  of ADOS, more and more catalog blocks can be allocated,
 *		  so long as there is space for them.  This routine,
 *		  however, DOES NOT try to add more catalog entries than
 *		  were originally allocated in the directory track.  It
 *		  may be nice in the future to make this possible, but with
 *		  normally 15 tracks of 7 entries (105 files) there doesn't
 *		  seem to be the need to worry about it.
 *		- the incoming filename is in unix format
 *		- this routine DOES NOT check to see if the file exists.
 *		  You should do that BEFORE calling this routine.
 ************************************************************************/
int
ados_catalog_new(struct adosfs *adosfs, char *name)
{
    int			i;
    char		*ados_name;
    ados_file_entry	file_entry;

    ados_name = ados_catalog_translate_in(name, &file_entry.type);

    strncpy(file_entry.name,ados_name,ADOS_FILE_NAME_LENGTH);

    file_entry.locked = FALSE;
    file_entry.length = 0;
    file_entry.deleted = FALSE;

    if (!ados_vtoc_sector_allocate(adosfs, &(file_entry.ts))) {
	return(FALSE);			/* no space for the first ts list	*/
    }

    /* note that the sector is allocated but not written	*/

    return(ados_catalog_add_file(adosfs->catalog, &file_entry));
}

/************************************************************************
 * NAME:	ados_catalog_lookup()
 *
 * DESCR:	Looks up the given unix-style filename in the catalog.
 *
 * ARGS:	
 *
 * RETURNS:	a pointer to the file_entry in which it is contained,
 *		or NULL if it was not found.
 *
 * NOTES:	- the extension in the filename is ignored for lookup.
 *		  That is, files will match independent upon type/extension.
 ************************************************************************/
ados_file_entry *
ados_catalog_lookup(struct adosfs *adosfs, char *name)
{
    int			i;
    char		*ados_name;
    ados_file_type	filetype;

    ados_name = ados_catalog_translate_in(name, &filetype);

    for (i=0; i < adosfs->catalog->index; i++) {
	if (strncasecmp(adosfs->catalog->files[i].name,ados_name, ADOS_FILE_NAME_LENGTH) == 0) {
	    return(&(adosfs->catalog->files[i]));
	}
    }
    return(NULL);
}

/************************************************************************
 * NAME:	ados_catalog_enumerate()
 *
 * DESCR:	Begin a walk through the catalog of a file-system.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
void
ados_catalog_enumerate(ados_catalog_cursor *cursor)
{	
    cursor->index = 0;
}

/************************************************************************
 * NAME:	ados_catalog_next()
 *
 * DESCR:	Return the (first) next catalog entry.  The name is
 *		returned.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
char *
ados_catalog_next(ados_catalog_cursor *cursor)
{
}

ados_catalog_report(ados_catalog *catalog, int verosity)
{
    int		i, total = 0;

    printf("D Type Size Filename                    \n");
    printf("- ---- ---- ----------------------------\n");

    for (i=0; i < catalog->index; i++) {
	printf("%s %4s %4d %*s\n", 
	       (catalog->files[i].deleted)?"X":" ",
	       ados_catalog_map_filetype(catalog->files[i].type),
	       catalog->files[i].length,
	       ADOS_FILE_NAME_LENGTH,
	       catalog->files[i].name);
	if (!catalog->files[i].deleted) {
	    total += catalog->files[i].length;
	}
    }

    printf("       ----\n");
    printf("       %4d\n",total);
}
       
